import express from 'express';
import { getAuth } from 'firebase-admin/auth';
import { authenticateToken, authorizeRole } from '../middleware/auth.js';

const router = express.Router();

// CRUD operations for users, accessible only by Administrators
router.use(authenticateToken); 

// Use a middleware to initialize the firebaseAuth instance for each request
router.use((req, res, next) => {
  req.firebaseAuth = getAuth();
  next();
});

// GET current user's details
router.get('/me', async (req, res) => {
  try {
    const uid = req.user.uid; // req.user is set by authenticateToken middleware
    const userRecord = await req.firebaseAuth.getUser(uid);

    // Extract custom claims and map them to the response
    const user = {
      uid: userRecord.uid,
      email: userRecord.email,
      role: userRecord.customClaims?.role || 'Operador',
      division: userRecord.customClaims?.division || 'N/A',
      subdivision: userRecord.customClaims?.subdivision || 'N/A',
      disabled: userRecord.disabled,
      creationTime: userRecord.metadata.creationTime,
      lastSignInTime: userRecord.metadata.lastSignInTime,
    };
    res.json(user);
  } catch (error) {
    console.error('Error fetching current user:', error);
    res.status(500).json({ error: 'Failed to fetch user data' });
  }
});

// GET all users
// GET all users (accessible to any authenticated user)
router.get('/', async (req, res) => {
  try {
    const listUsersResult = await req.firebaseAuth.listUsers();
    const users = listUsersResult.users.map(userRecord => ({
      uid: userRecord.uid,
      email: userRecord.email,
      displayName: userRecord.displayName,
      role: userRecord.customClaims?.role || 'Operador',
      division: userRecord.customClaims?.division || 'N/A',
      subdivision: userRecord.customClaims?.subdivision || 'N/A',
      disabled: userRecord.disabled,
      creationTime: userRecord.metadata.creationTime,
      lastSignInTime: userRecord.metadata.lastSignInTime,
    }));
    res.json(users);
  } catch (error) {
    console.error('Error fetching users:', error);
    res.status(500).json({ error: 'Failed to fetch users' });
  }
});
// POST create a new user
router.post('/', authorizeRole('Administrador'), async (req, res) => {
  try {
    const { email, password, role, division, subdivision } = req.body;
    
    // Ensure only admins can create new users, and can't create other admins
    if (role === 'Administrador' && req.user.role !== 'Super Administrador') {
      return res.status(403).json({ error: 'Permission denied: Cannot create an administrator account.' });
    }

    const userRecord = await req.firebaseAuth.createUser({
      email,
      password,
      displayName: req.user.displayName, // Or a name passed in the request body
    });

    // Set custom claims for the new user
    await req.firebaseAuth.setCustomUserClaims(userRecord.uid, { role, division, subdivision });

    res.status(201).json({ 
      uid: userRecord.uid, 
      email: userRecord.email, 
      role, 
      division, 
      subdivision 
    });
  } catch (error) {
    console.error('Error creating user:', error);
    res.status(500).json({ error: 'Failed to create user' });
  }
});

// PUT update a user (e.g., their role, or disable/enable them)
router.put('/:uid', authorizeRole('Administrador'), async (req, res) => {
  try {
    const { uid } = req.params;
    const { role, division, subdivision, disabled } = req.body;
    
    const userToUpdate = await req.firebaseAuth.getUser(uid);

    // Only update custom claims if they are provided
    const newClaims = {
      ...userToUpdate.customClaims,
      ...(role && { role }),
      ...(division && { division }),
      ...(subdivision && { subdivision }),
    };

    if (Object.keys(newClaims).length > 0) {
      await req.firebaseAuth.setCustomUserClaims(uid, newClaims);
    }

    if (disabled !== undefined) {
      await req.firebaseAuth.updateUser(uid, { disabled });
    }
    
    const updatedUser = await req.firebaseAuth.getUser(uid);
    res.json({
      uid: updatedUser.uid,
      email: updatedUser.email,
      role: updatedUser.customClaims?.role,
      division: updatedUser.customClaims?.division,
      subdivision: updatedUser.customClaims?.subdivision,
      disabled: updatedUser.disabled,
    });
  } catch (error) {
    console.error('Error updating user:', error);
    res.status(500).json({ error: 'Failed to update user' });
  }
});

// DELETE a user
router.delete('/:uid', authorizeRole('Administrador'), async (req, res) => {
  try {
    const { uid } = req.params;
    await req.firebaseAuth.deleteUser(uid);
    res.json({ message: 'User deleted successfully' });
  } catch (error) {
    console.error('Error deleting user:', error);
    res.status(500).json({ error: 'Failed to delete user' });
  }
});

export default router;