import express from 'express';
import { getAuth } from 'firebase-admin/auth';

const router = express.Router();

router.post('/', async (req, res) => {
  try {
    const firebaseAuth = getAuth();
    
    const { email, password, role, division, subdivision, displayName } = req.body;

    // Security check: Only allow this endpoint to be used if there are no 'Super Administrador' or 'Administrador' users yet.
    const allUsers = await firebaseAuth.listUsers();
    const hasAdmin = allUsers.users.some(
      user => user.customClaims?.role === 'Super Administrador' || user.customClaims?.role === 'Administrador'
    );
    
    if (hasAdmin) {
      return res.status(403).json({ error: 'Permission denied. An administrator already exists. This endpoint is for initial setup only.' });
    }

    // Validate the role
    const validRoles = ['Administrador', 'Super Administrador'];
    if (!validRoles.includes(role)) {
      return res.status(400).json({ error: 'Invalid role specified. Must be "Administrador" or "Super Administrador".' });
    }

    // FIX: Instead of creating a new user, get the user record that was just created by the frontend
    const userRecord = await firebaseAuth.getUserByEmail(email);

    // Set the custom claims for the new user
    await firebaseAuth.setCustomUserClaims(userRecord.uid, { role, division, subdivision });

    res.status(201).json({
      message: `${role} account created successfully.`,
      uid: userRecord.uid,
      email: userRecord.email,
    });

  } catch (error) {
    console.error('Error creating initial admin user:', error);
    if (error.code === 'auth/user-not-found') {
      res.status(404).json({ error: 'User not found. This should not happen if the user was just created by the frontend.' });
    } else {
      res.status(500).json({ error: 'Failed to create initial admin user.' });
    }
  }
});

export default router;