import express from 'express';
import multer from 'multer';
import path from 'path';
import Point from '../models/Point.js';
import { authenticateToken, authorizeRole } from '../middleware/auth.js';

const router = express.Router();

// Configure Multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/');
  },
  filename: (req, file, cb) => {
    cb(null, `${file.fieldname}-${Date.now()}${path.extname(file.originalname)}`);
  }
});
const upload = multer({ storage: storage });
// Get all points with optional filtering and search
router.get('/', authenticateToken, async (req, res) => {
  try {
    const { type, search, division, subdivision, page = 1, limit = 100 } = req.query;

    let query = {};
    
    // Filter by type, division, subdivision using their IDs
    if (type && type !== 'all') {
      query.type = type;
    }
    if (division && division !== 'all') {
      query.division = division;
    }
    if (subdivision && subdivision !== 'all') {
      query.subdivision = subdivision;
    }

    // Search by name or description
    if (search) {
      query.$or = [
        { name: { $regex: search, $options: 'i' } },
        { description: { $regex: search, $options: 'i' } }
      ];
    }

    const points = await Point.find(query)
      .populate('type', 'name') // Populate the 'type' field, returning only the 'name'
      .populate('division', 'name') // Populate the 'division' field
      .populate('subdivision', 'name') // Populate the 'subdivision' field
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ createdAt: -1 });

    const total = await Point.countDocuments(query);

    res.json({
      points,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Error fetching points:', error);
    res.status(500).json({ error: 'Failed to fetch points' });
  }
});

// Get single point by ID
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const point = await Point.findById(req.params.id)
      .populate('type', 'name')
      .populate('division', 'name')
      .populate('subdivision', 'name');
    
    if (!point) {
      return res.status(404).json({ error: 'Point not found' });
    }
    
    res.json(point);
  } catch (error) {
    console.error('Error fetching point:', error);
    res.status(500).json({ error: 'Failed to fetch point' });
  }
});


// POST Endpoint (Create)
router.post('/', authenticateToken, authorizeRole('Administrador'), upload.array('photos'), async (req, res) => {
  try {
    console.log('Received files:', req.files); // Log array of files
    console.log('Form fields:', req.body); // Log other form fields
    const { name, description, type, coordinates, division, subdivision } = req.body;

    if (!name || !description || !type || !coordinates || !division || !subdivision) {
      return res.status(400).json({ 
        error: 'Missing required fields: name, description, type, coordinates, division, subdivision' 
      });
    }

    const parsedCoordinates = typeof coordinates === 'string' ? JSON.parse(coordinates) : coordinates;

    const photos = [];
    if (req.files && Array.isArray(req.files)) {
      req.files.forEach((file, index) => {
        const photoUrl = `https://map-server.whirelab.com/${file.path.replace(/\\/g, '/')}`;
        const caption = req.body[`caption_${index}`] || file.originalname; // Use caption_${index} or fallback
        photos.push({ url: photoUrl, caption });
      });
    }

    const point = new Point({
      name: name.trim(),
      description: description.trim(),
      type,
      coordinates: {
        latitude: parseFloat(parsedCoordinates.latitude),
        longitude: parseFloat(parsedCoordinates.longitude)
      },
      photos,
      division,
      subdivision,
      createdBy: req.user.uid,
      updatedBy: req.user.uid
    });

    const savedPoint = await point.save();
    res.status(201).json(savedPoint);
  } catch (error) {
    console.error('Error creating point:', error);
    if (error.name === 'ValidationError') {
      return res.status(400).json({ 
        error: 'Validation error', 
        details: Object.values(error.errors).map(e => e.message)
      });
    }
    res.status(500).json({ error: 'Failed to create point' });
  }
});
// PUT Endpoint (Update)
router.put('/:id', authenticateToken, authorizeRole('Administrador'), upload.array('photos'), async (req, res) => {
  try {
    const { name, description, type, coordinates, division, subdivision, existingPhotos } = req.body;

    const updateData = { updatedBy: req.user.uid };

    if (name) updateData.name = name.trim();
    if (description) updateData.description = description.trim();
    if (type) updateData.type = type;
    if (coordinates) {
      const parsedCoordinates = typeof coordinates === 'string' ? JSON.parse(coordinates) : coordinates;
      updateData.coordinates = {
        latitude: parseFloat(parsedCoordinates.latitude),
        longitude: parseFloat(parsedCoordinates.longitude)
      };
    }
    if (division) updateData.division = division;
    if (subdivision) updateData.subdivision = subdivision;

    const newPhotos = [];
    if (req.files && Array.isArray(req.files)) {
      req.files.forEach((file, index) => {
        const photoUrl = `https://map-server.whirelab.com/${file.path.replace(/\\/g, '/')}`;
        const caption = req.body[`caption_${index}`] || file.originalname; // Use caption_${index} or fallback
        newPhotos.push({ url: photoUrl, caption });
      });
    }

    // Combine existing photos (if any) with the newly uploaded ones
    const parsedExistingPhotos = existingPhotos ? JSON.parse(existingPhotos) : [];
    updateData.photos = [...parsedExistingPhotos, ...newPhotos];

    const point = await Point.findByIdAndUpdate(
      req.params.id,
      updateData,
      { 
        new: true, 
        runValidators: true 
      }
    );

    if (!point) {
      return res.status(404).json({ error: 'Point not found' });
    }

    res.json(point);
  } catch (error) {
    console.error('Error updating point:', error);
    if (error.name === 'ValidationError') {
      return res.status(400).json({ 
        error: 'Validation error', 
        details: Object.values(error.errors).map(e => e.message)
      });
    }
    res.status(500).json({ error: 'Failed to update point' });
  }
});
// Delete point (only for admin)
router.delete('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const point = await Point.findByIdAndDelete(req.params.id);
    
    if (!point) {
      return res.status(404).json({ error: 'Point not found' });
    }
    
    res.json({ message: 'Point deleted successfully' });
  } catch (error) {
    console.error('Error deleting point:', error);
    res.status(500).json({ error: 'Failed to delete point' });
  }
});

// // Get point types
// router.get('/meta/types', (req, res) => {
//   const types = ['Armazém', 'PTS', 'Sucursal', 'Posto EDM', 'Parceiro'];
//   res.json(types);
// });

export default router;