import mongoose from 'mongoose';

const pointSchema = new mongoose.Schema({
  name: {
    type: String,
    required: [true, 'Point name is required'],
    trim: true,
    maxlength: [100, 'Name cannot exceed 100 characters']
  },
  description: {
    type: String,
    required: [true, 'Description is required'],
    trim: true,
    maxlength: [500, 'Description cannot exceed 500 characters']
  },
  photos: [{
    url: { type: String, required: true },
    caption: { type: String, trim: true, maxlength: 200 }
  }],
  type: {
    type: mongoose.Schema.Types.ObjectId, // Change to ObjectId
    ref: 'PointType',                    // Reference the new model
    required: [true, 'Point type is required']
  },
  division: {
    type: mongoose.Schema.Types.ObjectId, // Change to ObjectId
    ref: 'Division',                      // Reference the new model
    required: [true, 'Regional division is required']
  },
  subdivision: {
    type: mongoose.Schema.Types.ObjectId, // Change to ObjectId
    ref: 'Subdivision',                   // Reference the new model
    required: [true, 'Subdivision is required']
  },
  coordinates: {
    latitude: {
      type: Number,
      required: [true, 'Latitude is required'],
      min: [-90, 'Latitude must be between -90 and 90'],
      max: [90, 'Latitude must be between -90 and 90']
    },
    longitude: {
      type: Number,
      required: [true, 'Longitude is required'],
      min: [-180, 'Longitude must be between -180 and 180'],
      max: [180, 'Longitude must be between -180 and 180']
    }
  },
  createdBy: {
    type: String,
    required: [true, 'Creator ID is required']
  },
  updatedBy: {
    type: String,
    required: [true, 'Updater ID is required']
  }
}, {
  timestamps: true,
  toJSON: { virtuals: true },
  toObject: { virtuals: true }
});

// Index for geospatial queries
pointSchema.index({ 'coordinates.latitude': 1, 'coordinates.longitude': 1 });

// Index for text search
pointSchema.index({ name: 'text', description: 'text' });

// Virtual for icon based on type
pointSchema.virtual('icon').get(function() {
  const iconMap = {
    'Armazém': 'warehouse',
    'PTS': 'zap',
    'Sucursal': 'building',
    'Posto EDM': 'map-pin',
    'Parceiro': 'handshake'
  };
  return iconMap[this.type] || 'map-pin';
});

export default mongoose.model('Point', pointSchema);