import admin from 'firebase-admin';
import { getStorage } from 'firebase-admin/storage';
import { v4 as uuidv4 } from 'uuid';

export const initializeFirebase = () => {
  try {
    const serviceAccount = {
      type: "service_account",
      project_id: process.env.FIREBASE_PROJECT_ID,
      private_key_id: process.env.FIREBASE_PRIVATE_KEY_ID,
      private_key: process.env.FIREBASE_PRIVATE_KEY?.replace(/\\n/g, '\n'),
      client_email: process.env.FIREBASE_CLIENT_EMAIL,
      client_id: process.env.FIREBASE_CLIENT_ID,
      auth_uri: process.env.FIREBASE_AUTH_URI,
      token_uri: process.env.FIREBASE_TOKEN_URI,
    };

    admin.initializeApp({
      credential: admin.credential.cert(serviceAccount),
      projectId: process.env.FIREBASE_PROJECT_ID,
      storageBucket: process.env.FIREBASE_STORAGE_BUCKET,
    });

    console.log('Firebase Admin initialized successfully');
  } catch (error) {
    console.error('Firebase Admin initialization error:', error);
  }
};

export const verifyFirebaseToken = async (token) => {
  try {
    const decodedToken = await admin.auth().verifyIdToken(token);
    return decodedToken;
  } catch (error) {
    throw new Error('Invalid Firebase token');
  }
};

export const uploadFileToFirebaseStorage = async (fileBuffer, filePath, metadata = {}) => {
  try {
    const bucket = getStorage().bucket();
    const file = bucket.file(filePath);

    // Create a write stream to upload the file
    const stream = file.createWriteStream({
      metadata: {
        contentType: metadata.contentType || 'application/octet-stream',
        metadata: {
          firebaseStorageDownloadTokens: uuidv4(),
        },
      },
    });

    // Pipe the file buffer to the stream
    await new Promise((resolve, reject) => {
      stream.end(fileBuffer, (err) => {
        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });

    // Get the public URL for the uploaded file
    const publicUrl = `https://storage.googleapis.com/${bucket.name}/${file.name}`;

    return publicUrl;
  } catch (error) {
    console.error('Error uploading file to Firebase Storage:', error);
    throw new Error('Failed to upload file to Firebase Storage');
  }
};